/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mx_auto_config.h"
#include "myriexpress.h"
#if !MX_OS_WINNT
#include <unistd.h>
#include <sys/time.h>
#else
#include "mx_uni.h"
#endif
#include "test_common.h"

#define FILTER     0x12345
#define MATCH_VAL  0xabcdef
#define DFLT_EID   1

void
usage()
{
  fprintf(stderr, "Usage: mx_unexpected_test [args]\n");
  fprintf(stderr, "-n nic_id - local NIC ID [MX_ANY_NIC]\n");
  fprintf(stderr, "-b board_id - local Board ID [MX_ANY_NIC]\n");
  fprintf(stderr, "-e local_eid - local endpoint ID [%d]\n", DFLT_EID);
  fprintf(stderr, "-r remote_eid - remote endpoint ID [%d]\n", DFLT_EID);
  fprintf(stderr, "-d hostname - destination hostname, required for sender\n");
  fprintf(stderr, "-h - help\n");
}

void
test(int sender, mx_endpoint_t ep, mx_endpoint_addr_t dest)
{
  mx_segment_t seg[3];
  char buf[3][80];
  mx_request_t req[3];
  mx_status_t s;
  uint32_t result;

  seg[0].segment_ptr = buf[0];
  seg[1].segment_ptr = buf[1];
  seg[2].segment_ptr = buf[2];

  if (sender) {
    strcpy(buf[0], "foo");
    strcpy(buf[1], "bar");
    strcpy(buf[2], "baz");
    seg[0].segment_length = strlen(buf[0])+1;
    seg[1].segment_length = strlen(buf[1])+1;
    seg[2].segment_length = strlen(buf[2])+1;
    mx_isend(ep, &seg[0], 1, dest, MAKE_MATCH(0, 2), NULL, &req[0]);
    mx_isend(ep, &seg[1], 1, dest, MAKE_MATCH(1, 1), NULL, &req[1]);
    mx_isend(ep, &seg[2], 1, dest, MAKE_MATCH(2, 0), NULL, &req[2]);
    mx_wait(ep, &req[2], MX_INFINITE, &s, &result);
    mx_wait(ep, &req[1], MX_INFINITE, &s, &result);
    mx_wait(ep, &req[0], MX_INFINITE, &s, &result);
  }
  else {
    strcpy(buf[0], "");
    strcpy(buf[1], "");
    strcpy(buf[2], "");
    seg[0].segment_length = 80;
    seg[1].segment_length = 80;
    seg[2].segment_length = 80;
    mx_irecv(ep, &seg[2], 1, MAKE_MATCH(2, 0), MX_MATCH_MASK_NONE, NULL, &req[2]);
    mx_wait(ep, &req[2], MX_INFINITE, &s, &result);
    mx_irecv(ep, &seg[1], 1, MAKE_MATCH(1, 1), MX_MATCH_MASK_NONE, NULL, &req[1]);
    mx_irecv(ep, &seg[0], 1, MAKE_MATCH(0, 2), MX_MATCH_MASK_NONE, NULL, &req[0]);
    mx_wait(ep, &req[0], MX_INFINITE, &s, &result);
    mx_wait(ep, &req[1], MX_INFINITE, &s, &result);
  }
}

int
main(int argc, char **argv)
{
  mx_return_t rc;
  mx_endpoint_t ep;
  uint64_t nic_id;
  uint32_t board_id;
  uint16_t my_eid;
  uint64_t his_nic_id;
  uint32_t filter;
  uint16_t his_eid;
  mx_endpoint_addr_t his_addr;
  char *rem_host;
  int c;
  extern char *optarg;

  /* set up defaults */
  rem_host = NULL;
  filter = FILTER;
  my_eid = DFLT_EID;
  his_eid = DFLT_EID;
  board_id = MX_ANY_NIC;

  mx_init();
  mx_set_error_handler(MX_ERRORS_RETURN);
  while ((c = getopt(argc, argv, "hd:e:n:b:r:")) != EOF) switch(c) {
  case 'd':
    rem_host = optarg;
    break;
  case 'e':
    my_eid = atoi(optarg);
    break;
  case 'n':
    sscanf(optarg, "%"SCNx64, &nic_id);
    rc = mx_nic_id_to_board_number(nic_id, &board_id);
    if (rc != MX_SUCCESS) {
      fprintf(stderr, "nic_id %012"PRIx64" can't be found\n", nic_id);
      exit(1);
    }
    break;
  case 'b':
    board_id = atoi(optarg);
    break;
  case 'r':
    his_eid = atoi(optarg);
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }

  rc = mx_open_endpoint(board_id, my_eid, filter, NULL, 0, &ep);
  if (rc != MX_SUCCESS) {
    fprintf(stderr, "mx_open_endpoint failed: %s\n", mx_strerror(rc));
    goto abort_with_init;
  }
  
  /* If no host, we are receiver */
  if (rem_host == NULL) {
    
    printf("Starting unexpected receiver\n");
    test(0, ep, his_addr);

  /* remote hostname implies we are sender */
  } else {

    /* get address of destination */
    rc = mx_hostname_to_nic_id(rem_host, &his_nic_id);
    if (rc != MX_SUCCESS) {
      fprintf(stderr, "Error getting remote NIC ID: %s\n", mx_strerror(rc));
      goto abort_with_endpoint;
    }
    rc = mx_connect(ep, his_nic_id, his_eid, filter, MX_INFINITE, &his_addr);
    if (rc != MX_SUCCESS) {
      fprintf(stderr, "Error composing remote endpt: %s\n", mx_strerror(rc));
      goto abort_with_endpoint;
    }

    printf("Starting unexpected send to host %s\n", rem_host);

    /* start up the sender */
    test(1, ep, his_addr);
  }

  
 abort_with_endpoint:
  mx_close_endpoint(ep);
 abort_with_init:
  mx_finalize();

  exit(rc != MX_SUCCESS);
}
